/*****************************************************************************
*  LPC11U6x_E6x PWM 4ch program
*
* Show simple four-channel PWM generation.
* Uses the unified 32-bit timer mode to generate single-edge aligned outputs.
* Channels can have different polarity. Demo state machine has been configured for positive pulses at SCTOUT_0/1
  and negative pulses at SCTOUT_2/3.
* SCTIN_0 is used a #ABORT input. If low, forces the outputs to their idle states, halts the timer, and generates
  an interrupt.
* An interrupt can be generated after each cycle.
*****************************************************************************/
#include "board.h"

#define pwm_val1        (400000)                           				// duty cycle PWM1
#define pwm_val2        (500000)                           				// duty cycle PWM2
#define pwm_val3        (100000)                           				// duty cycle PWM3
#define pwm_val4        (900000)                           				// duty cycle PWM4
#define pwm_cycle       (1000000)

volatile uint32_t numPwmCycles;
volatile int      pwmAborted;
volatile int      pwmPhase;

void SCT1_Init(void);

void SCT1_Init(void)
{

	Chip_SCT_Init(LPC_SCT1);			                   				// enable the SCT2 clock

	Chip_SCT_Config(LPC_SCT1, 	SCT_CONFIG_32BIT_COUNTER |				// unified timers,
								SCT_CONFIG_AUTOLIMIT_L   );				// auto limit


	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_0,	pwm_cycle);						// match 0 on PWM cycle
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_0, pwm_cycle);

	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_1,	pwm_val1);						// match 1 on val1 (PWM1)
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_1, pwm_val1);


	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_2,	pwm_val2);						// match 2 on val2 (PWM2)
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_2, pwm_val2);

	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_3,	pwm_val3);						// match 3 on val3 (PWM3)
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_3, pwm_val3);

	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_4,	pwm_val4);						// match 4 on val4 (PWM4)
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_4, pwm_val4);


	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_0, ENABLE_ALL_STATES);	 	// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0 |
                                                                        SCT_COMBMODE_MATCH ));  // match 0 (pwm_cycle) only condition

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_1, ENABLE_ALL_STATES);	 	// event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH1 |
                                                                        SCT_COMBMODE_MATCH )); 	// match 1 (pwm_val1) only condition

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_2, ENABLE_ALL_STATES);	 	// event 2 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_2, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH2	|
                                                                        SCT_COMBMODE_MATCH )); 	// match 2 (pwm_val2) only condition

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_3, ENABLE_ALL_STATES);	 	// event 3 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_3,(CHIP_SCT_EVENTCTRL_T) (  SCT_EVECTRL_MATCH3 |
                                                                        SCT_COMBMODE_MATCH )); 	// match 3 (pwm_val3) only condition

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_4, ENABLE_ALL_STATES);	 	// event 4 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_4, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH4 |
                                                                        SCT_COMBMODE_MATCH )); 	// match 4 (pwm_val4) only condition

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_5, ENABLE_ALL_STATES);	 	// event 5 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_5, (CHIP_SCT_EVENTCTRL_T) ( SCT_IOCOND_LOW    |
                                                                        SCT_COMBMODE_IO   )); 	// IN_0 LOW only condition


	Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_0, SCT_EVT_0 );  			 // event 0       sets  OUT0 (PWM1)
	Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_0, (CHIP_SCT_EVENT_T) ( SCT_EVT_1 | SCT_EVT_5)); // event 1 and 5 clear OUT0 (PWM1)

	Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_1, SCT_EVT_0 );  			 // event 0       sets  OUT1 (PWM2)
	Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_1, (CHIP_SCT_EVENT_T) ( SCT_EVT_2 | SCT_EVT_5)); // event 2 and 5 clear OUT1 (PWM2)

	Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_2, (CHIP_SCT_EVENT_T) ( SCT_EVT_3 | SCT_EVT_5));  // event 3 and 5 set   OUT2 (PWM3)
	Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_2, SCT_EVT_0); 			// event 0       clear OUT2 (PWM3)

	Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_3, (CHIP_SCT_EVENT_T) ( SCT_EVT_4 | SCT_EVT_5));  // event 4 and 5 set   OUT3 (PWM4)
	Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_3, SCT_EVT_0); 			// event 0       clear OUT3 (PWM4)

	Chip_SCT_Output(LPC_SCT1,0x0C);                       				// default clear OUT0/1 and set OUT2/3

	Chip_SCT_SetConflictResolution(LPC_SCT1, 0x00, 0x5A);  				// conflict resolution: Inactive state takes precedence
                                                               	   	   	// SCT2_OUT0/1: Inactive state low
                                                               	   	   	// SCT2_OUT2/3: Inactive state high

    LPC_SCT1->HALT_L            = (1 << 5);                				// event 5 will halt the timer
    LPC_SCT1->LIMIT_L           = (1 << 5);                				// event 5 will limit the timer

    Chip_SCT_EnableEventInt(LPC_SCT1,(CHIP_SCT_EVENT_T) ( SCT_EVT_0 | SCT_EVT_5));			// event 0 and 5 will generate an irq

    NVIC_EnableIRQ(SCT0_1_IRQn);                           				// enable SCT interrupt


    Chip_SCT_ClearControl(LPC_SCT1,SCT_CTRL_HALT_L);					// start timer

}


void SCT0_1_IRQHandler(void)
{
  uint32_t status = Chip_SCT_GetEventFlag(LPC_SCT1);

    if (status & SCT_EVE_0)                                	// event 0 irq?
    {
        ++numPwmCycles;                                    	// interrupt once per PWM cycle
    }
    if (status & SCT_EVE_5)                                 // event 5 irq?
    {
        pwmAborted = 1;                                    	// Abort interrupt
    }
    LPC_SCT1->EVFLAG = status;                             // clear interrupt flag
}

int main (void) 
{
	uint32_t lastCycles;

	SystemCoreClockUpdate();
	Board_Init();

	Chip_Clock_SetSysClockDiv(4);

	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_IOCON);				// enable clock for IOCON


	Chip_IOCON_PinMuxSet(LPC_IOCON, 2, 8 ,0x00000081);				// P2_8  is SCT1_IN0 used as ABORT
	Chip_IOCON_PinMuxSet(LPC_IOCON, 2,16 ,0x00000081);				// P2_16 is SCT1_OUT0 = green trace  = PWM1
	Chip_IOCON_PinMuxSet(LPC_IOCON, 2,17 ,0x00000081);				// P2_17 is SCT1_OUT1 = red trace    = PWM2
	Chip_IOCON_PinMuxSet(LPC_IOCON, 2,18 ,0x00000081);            	// P2_18 is SCT1_OUT2 = yellow trace = PWM3
	Chip_IOCON_PinMuxSet(LPC_IOCON, 2,19 ,0x00000081);              // P2_19 is SCT1_OUT3 = blue trace   = PWM4

    SCT1_Init();                                   					// Initialize SCT


    while (1)                                              								// loop forever
    {
		if (numPwmCycles != lastCycles)
		  {
			lastCycles = numPwmCycles;
			if ((lastCycles % 5) == 0)                 										// every few PWM cycles change the duty cycles
			{
				Chip_SCT_SetClrControl(LPC_SCT1, SCT_CONFIG_NORELOADL_U, ENABLE);			// NORELOAD_L (U)
			   if (pwmPhase == 0)
			   {
				  Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_1,  200000);
				  Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_3,  700000);
			   }
			   else
			   {
				  Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_1,  950000);
				  LPC_SCT1->MATCHREL[3].U = Chip_SCT_GetMatchReload(LPC_SCT1, SCT_MATCH_0);	// duty cycle 0 (test conflict resolution)
			   }
			   	   Chip_SCT_SetClrControl(LPC_SCT1, SCT_CONFIG_NORELOADL_U, DISABLE);		// NORELOAD_L (U)
			   	   ++pwmPhase;
			   if (pwmPhase > 1)
			   {
				   pwmPhase = 0;
			   }
			}
		   }
		   if (pwmAborted)          /* Demo: Poll ABORT input, and restart timer if abort condition has gone. */
		   {
			   while (!(LPC_SCT1->INPUT & (1 << 0))) ;

			   Chip_SCT_ClearControl(LPC_SCT1,SCT_CTRL_HALT_L);		// start timer
			   pwmAborted = 0;
		   }

		__WFI();
	 }

}

